<?php
/**
 * Breadcrumb Trail - A breadcrumb menu script for WordPress.
 *
 * Breadcrumb Trail is a script for showing a breadcrumb trail for any type of page.  It tries to
 * anticipate any type of structure and display the best possible trail that matches your site's
 * permalink structure.  While not perfect, it attempts to fill in the gaps left by many other
 * breadcrumb scripts.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * @package   BreadcrumbTrail
 * @version   1.1.0
 * @author    Justin Tadlock <justin@justintadlock.com>
 * @copyright Copyright (c) 2008 - 2017, Justin Tadlock
 * @link      https://themehybrid.com/plugins/breadcrumb-trail
 * @license   http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

/**
 * Shows a breadcrumb for all types of pages.  This is a wrapper function for the Travel_Tourist_Breadcrumb_Trail class,
 * which should be used in theme templates.
 *
 * @since  0.1.0
 * @access public
 * @param  array $travel_tourist_args Arguments to pass to Travel_Tourist_Breadcrumb_Trail.
 * @return void
 */
function travel_tourist_breadcrumbs_trail( $travel_tourist_args = array() ) {

    $travel_tourist_breadcrumb = apply_filters( 'travel_tourist_breadcrumbs_trail_object', null, $travel_tourist_args );

    if ( ! is_object( $travel_tourist_breadcrumb ) )
        $travel_tourist_breadcrumb = new Travel_Tourist_Breadcrumb_Trail( $travel_tourist_args );

    return $travel_tourist_breadcrumb->trail();
}

/**
 * Creates a breadcrumbs menu for the site based on the current page that's being viewed by the user.
 *
 * @since  0.6.0
 * @access public
 */
class Travel_Tourist_Breadcrumb_Trail {

    /**
     * Array of items belonging to the current breadcrumb trail.
     *
     * @since  0.1.0
     * @access public
     * @var    array
     */
    public $items = array();

    /**
     * Arguments used to build the breadcrumb trail.
     *
     * @since  0.1.0
     * @access public
     * @var    array
     */
    public $travel_tourist_args = array();

    /**
     * Array of text labels.
     *
     * @since  1.0.0
     * @access public
     * @var    array
     */
    public $travel_tourist_labels = array();

    /**
     * Array of post types (key) and taxonomies (value) to use for single post views.
     *
     * @since  1.0.0
     * @access public
     * @var    array
     */
    public $travel_tourist_post_taxonomy = array();

    public $labels;
    public $post_taxonomy;
	public $args;

    /* ====== Magic Methods ====== */

    /**
     * Magic method to use in case someone tries to output the layout object as a string.
     * We'll just return the trail HTML.
     *
     * @since  1.0.0
     * @access public
     * @return string
     */
    public function __toString() {
        return $this->trail();
    }

    /**
     * Sets up the breadcrumb trail properties.  Calls the `Travel_Tourist_Breadcrumb_Trail::add_items()` method
     * to creat the array of breadcrumb items.
     *
     * @since  0.6.0
     * @access public
     * @param  array   $travel_tourist_args  {
     *     @type string    $container      Container HTML element. nav|div
     *     @type string    $before         String to output before breadcrumb menu.
     *     @type string    $after          String to output after breadcrumb menu.
     *     @type string    $browse_tag     The HTML tag to use to wrap the "Browse" header text.
     *     @type string    $list_tag       The HTML tag to use for the list wrapper.
     *     @type string    $travel_tourist_item_tag       The HTML tag to use for the item wrapper.
     *     @type bool      $show_on_front  Whether to show when `is_front_page()`.
     *     @type bool      $travel_tourist_network        Whether to link to the network main site (multisite only).
     *     @type bool      $show_title     Whether to show the title (last item) in the trail.
     *     @type bool      $show_browse    Whether to show the breadcrumb menu header.
     *     @type array     $travel_tourist_labels         Text labels. @see Travel_Tourist_Breadcrumb_Trail::set_labels()
     *     @type array     $travel_tourist_post_taxonomy  Taxonomies to use for post types. @see Travel_Tourist_Breadcrumb_Trail::set_post_taxonomy()
     *     @type bool      $echo           Whether to print or return the breadcrumbs.
     * }
     * @return void
     */
    public function __construct( $travel_tourist_args = array() ) {

        $travel_tourist_defaults = array(
            'container'       => 'nav',
            'before'          => '',
            'after'           => '',
            'browse_tag'      => 'h2',
            'list_tag'        => 'ul',
            'item_tag'        => 'li',
            'show_on_front'   => true,
            'network'         => false,
            'show_title'      => true,
            'show_browse'     => false,
            'labels'          => array(),
            'post_taxonomy'   => array(),
            'echo'            => true
        );

        // Parse the arguments with the deaults.
        $this->args = apply_filters( 'travel_tourist_breadcrumbs_trail_args', wp_parse_args( $travel_tourist_args, $travel_tourist_defaults ) );

        // Set the labels and post taxonomy properties.
        $this->set_labels();
        $this->set_post_taxonomy();

        // Let's find some items to add to the trail!
        $this->add_items();
    }

    /* ====== Public Methods ====== */

    /**
     * Formats the HTML output for the breadcrumb trail.
     *
     * @since  0.6.0
     * @access public
     * @return string
     */
    public function trail() {

        // Set up variables that we'll need.
        $travel_tourist_breadcrumb    = '';
        $travel_tourist_item_count    = count( $this->items );
        $travel_tourist_item_position = 0;

        // Connect the breadcrumb trail if there are items in the trail.
        if ( 0 < $travel_tourist_item_count ) {

            // Add 'browse' label if it should be shown.
            if ( true === $this->args['show_browse'] ) {

                $travel_tourist_breadcrumb .= sprintf(
                    '<%1$s class="trail-browse">%2$s</%1$s>',
                    tag_escape( $this->args['browse_tag'] ),
                    $this->labels['browse']
                );
            }

            // Open the unordered list.
            $travel_tourist_breadcrumb .= sprintf(
                '<%s class="trail-items" itemscope itemtype="http://schema.org/BreadcrumbList">',
                tag_escape( $this->args['list_tag'] )
            );

            // Add the number of items and item list order schema.
            $travel_tourist_breadcrumb .= sprintf( '<meta name="numberOfItems" content="%d" />', absint( $travel_tourist_item_count ) );
            $travel_tourist_breadcrumb .= '<meta name="itemListOrder" content="Ascending" />';

            // Loop through the items and add them to the list.
            foreach ( $this->items as $travel_tourist_item ) {

                // Iterate the item position.
                ++$travel_tourist_item_position;

                // Check if the item is linked.
                preg_match( '/(<a.*?>)(.*?)(<\/a>)/i', $travel_tourist_item, $travel_tourist_matches );

                // Wrap the item text with appropriate itemprop.
                $travel_tourist_item = ! empty( $travel_tourist_matches ) ? sprintf( '%s<span itemprop="name">%s</span>%s', $travel_tourist_matches[1], $travel_tourist_matches[2], $travel_tourist_matches[3] ) : sprintf( '<span itemprop="name">%s</span>', $travel_tourist_item );

                // If viewing a single post.
                if ( is_singular() || is_page() ){

                    global $post;
                    $travel_tourist_link_item = get_permalink( $post->ID );

                }
                elseif ( is_archive() && ( !is_day() && !is_month() && !is_year() && !is_tag() ) ) {

                    $travel_tourist_category = get_queried_object();
                    $travel_tourist_link_item = get_category_link( $travel_tourist_category->term_id );

                } elseif ( is_day() ) {

                    $travel_tourist_link_item = get_day_link( get_the_time('Y'),get_the_time('m'),get_the_time('d') );

                } elseif ( is_month() ) {

                   $travel_tourist_link_item = get_month_link( get_the_time('Y'),get_the_time('m') );

                } elseif ( is_year() ) {

                    $travel_tourist_link_item = get_year_link( get_the_time('Y') );

                }elseif ( is_tag() ) {

                    $travel_tourist_tag_id = get_queried_object()->term_id;
                    $travel_tourist_link_item = get_tag_link( $travel_tourist_tag_id );

                } elseif ( is_author() ) {

                    $travel_tourist_link_item = get_author_posts_url( get_current_user_id() );

                }
                elseif ( is_search() ) {

                    $travel_tourist_link_item = home_url();

                }
                else{

                    $travel_tourist_link_item = home_url();

                }

                $travel_tourist_item = ! empty( $travel_tourist_matches )
                    ? preg_replace( '/(<a.*?)([\'"])>/i', '$1$2 itemprop=$2item$2>', $travel_tourist_item )
                    : sprintf( '<a href="'.esc_url( $travel_tourist_link_item ).'" itemprop="item">%s</a>', $travel_tourist_item );

                // Add list item classes.
                $travel_tourist_item_class = 'trail-item';

                if ( 1 === $travel_tourist_item_position && 1 < $travel_tourist_item_count )
                    $travel_tourist_item_class .= ' trail-begin';

                elseif ( $travel_tourist_item_count === $travel_tourist_item_position )
                    $travel_tourist_item_class .= ' trail-end';

                // Create list item attributes.
                $attributes = 'itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem" class="' . $travel_tourist_item_class . '"';

                // Build the meta position HTML.
                $meta = sprintf( '<meta itemprop="position" content="%s" />', absint( $travel_tourist_item_position ) );

                // Build the list item.
                $travel_tourist_breadcrumb .= sprintf( '<%1$s %2$s>%3$s%4$s</%1$s>', tag_escape( $this->args['item_tag'] ),$attributes, $travel_tourist_item, $meta );
            }

            // Close the unordered list.
            $travel_tourist_breadcrumb .= sprintf( '</%s>', tag_escape( $this->args['list_tag'] ) );

            // Wrap the breadcrumb trail.
            $travel_tourist_breadcrumb = sprintf(
                '<%1$s role="navigation" aria-label="%2$s" class="breadcrumb-trail" itemprop="breadcrumb">%3$s%4$s%5$s</%1$s>',
                tag_escape( $this->args['container'] ),
                esc_attr( $this->labels['aria_label'] ),
                $this->args['before'],
                $travel_tourist_breadcrumb,
                $this->args['after']
            );
        }

        // Allow developers to filter the breadcrumb trail HTML.
        $travel_tourist_breadcrumb = apply_filters( 'travel_tourist_breadcrumbs_trail', $travel_tourist_breadcrumb, $this->args );

        if ( false === $this->args['echo'] )
            return $travel_tourist_breadcrumb;

        echo $travel_tourist_breadcrumb;
    }

    /* ====== Protected Methods ====== */

    /**
     * Sets the labels property.  Parses the inputted labels array with the defaults.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function set_labels() {

        $travel_tourist_defaults = array(
            'browse'              => esc_html__( 'Browse:',                               'travel-tourist' ),
            'aria_label'          => esc_attr_x( 'Breadcrumbs', 'breadcrumbs aria label', 'travel-tourist' ),
            'home'                => esc_html__( 'Home',                                  'travel-tourist' ),
            'error_404'           => esc_html__( '404 Not Found',                         'travel-tourist' ),
            'archives'            => esc_html__( 'Archives',                              'travel-tourist' ),
            // Translators: %s is the search query.
            'search'              => esc_html__( 'Search results for: %s',                'travel-tourist' ),
            // Translators: %s is the page number.
            'paged'               => esc_html__( 'Page %s',                               'travel-tourist' ),
            // Translators: %s is the page number.
            'paged_comments'      => esc_html__( 'Comment Page %s',                       'travel-tourist' ),
            // Translators: Minute archive title. %s is the minute time format.
            'archive_minute'      => esc_html__( 'Minute %s',                             'travel-tourist' ),
            // Translators: Weekly archive title. %s is the week date format.
            'archive_week'        => esc_html__( 'Week %s',                               'travel-tourist' ),

            // "%s" is replaced with the translated date/time format.
            'archive_minute_hour' => '%s',
            'archive_hour'        => '%s',
            'archive_day'         => '%s',
            'archive_month'       => '%s',
            'archive_year'        => '%s',
        );

        $this->labels = apply_filters( 'travel_tourist_breadcrumbs_trail_labels', wp_parse_args( $this->args['labels'], $travel_tourist_defaults ) );
    }

    /**
     * Sets the `$travel_tourist_post_taxonomy` property.  This is an array of post types (key) and taxonomies (value).
     * The taxonomy's terms are shown on the singular post view if set.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function set_post_taxonomy() {

        $travel_tourist_defaults = array();

        // If post permalink is set to `%postname%`, use the `category` taxonomy.
        if ( '%postname%' === trim( get_option( 'permalink_structure' ), '/' ) )
            $travel_tourist_defaults['post'] = 'category';

        $this->post_taxonomy = apply_filters( 'travel_tourist_breadcrumbs_trail_post_taxonomy', wp_parse_args( $this->args['post_taxonomy'], $travel_tourist_defaults ) );
    }

    /**
     * Runs through the various WordPress conditional tags to check the current page being viewed.  Once
     * a condition is met, a specific method is launched to add items to the `$items` array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_items() {

        // If viewing the front page.
        if ( is_front_page() ) {
            $this->add_front_page_items();
        }

        // If not viewing the front page.
        else {

            // Add the network and site home links.
            $this->add_network_home_link();
            $this->add_site_home_link();

            // If viewing the home/blog page.
            if ( is_home() ) {
                $this->add_blog_items();
            }

            // If viewing a single post.
            elseif ( is_singular() ) {
                $this->add_singular_items();
            }

            // If viewing an archive page.
            elseif ( is_archive() ) {

                if ( is_post_type_archive() )
                    $this->add_post_type_archive_items();

                elseif ( is_category() || is_tag() || is_tax() )
                    $this->add_term_archive_items();

                elseif ( is_author() )
                    $this->add_user_archive_items();

                elseif ( get_query_var( 'minute' ) && get_query_var( 'hour' ) )
                    $this->add_minute_hour_archive_items();

                elseif ( get_query_var( 'minute' ) )
                    $this->add_minute_archive_items();

                elseif ( get_query_var( 'hour' ) )
                    $this->add_hour_archive_items();

                elseif ( is_day() )
                    $this->add_day_archive_items();

                elseif ( get_query_var( 'w' ) )
                    $this->add_week_archive_items();

                elseif ( is_month() )
                    $this->add_month_archive_items();

                elseif ( is_year() )
                    $this->add_year_archive_items();

                else
                    $this->add_default_archive_items();
            }

            // If viewing a search results page.
            elseif ( is_search() ) {
                $this->add_search_items();
            }

            // If viewing the 404 page.
            elseif ( is_404() ) {
                $this->add_404_items();
            }
        }

        // Add paged items if they exist.
        $this->add_paged_items();

        // Allow developers to overwrite the items for the breadcrumb trail.
        $this->items = array_unique( apply_filters( 'travel_tourist_breadcrumbs_trail_items', $this->items, $this->args ) );
    }

    /**
     * Gets front items based on $wp_rewrite->front.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_rewrite_front_items() {
        global $wp_rewrite;

        if ( $wp_rewrite->front )
            $this->add_path_parents( $wp_rewrite->front );
    }

    /**
     * Adds the page/paged number to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_paged_items() {

        // If viewing a paged singular post.
        if ( is_singular() && 1 < get_query_var( 'page' ) && true === $this->args['show_title'] )
            $this->items[] = sprintf( $this->labels['paged'], number_format_i18n( absint( get_query_var( 'page' ) ) ) );

        // If viewing a singular post with paged comments.
        elseif ( is_singular() && get_option( 'page_comments' ) && 1 < get_query_var( 'cpage' ) )
            $this->items[] = sprintf( $this->labels['paged_comments'], number_format_i18n( absint( get_query_var( 'cpage' ) ) ) );

        // If viewing a paged archive-type page.
        elseif ( is_paged() && true === $this->args['show_title'] )
            $this->items[] = sprintf( $this->labels['paged'], number_format_i18n( absint( get_query_var( 'paged' ) ) ) );
    }

    /**
     * Adds the network (all sites) home page link to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_network_home_link() {

        if ( is_multisite() && ! is_main_site() && true === $this->args['network'] )
            $this->items[] = sprintf( '<a href="%s" rel="home">%s</a>', esc_url( network_home_url() ), $this->labels['home'] );
    }

    /**
     * Adds the current site's home page link to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_site_home_link() {

        $travel_tourist_network = is_multisite() && ! is_main_site() && true === $this->args['network'];
        $travel_tourist_label   = $travel_tourist_network ? get_bloginfo( 'name' ) : $this->labels['home'];
        $rel     = $travel_tourist_network ? '' : ' rel="home"';

        $this->items[] = sprintf( '<a href="%s"%s>%s</a>', esc_url( user_trailingslashit( home_url() ) ), $rel, $travel_tourist_label );
    }

    /**
     * Adds items for the front page to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_front_page_items() {

        // Only show front items if the 'show_on_front' argument is set to 'true'.
        if ( true === $this->args['show_on_front'] || is_paged() || ( is_singular() && 1 < get_query_var( 'page' ) ) ) {

            // Add network home link.
            $this->add_network_home_link();

            // If on a paged view, add the site home link.
            if ( is_paged() )
                $this->add_site_home_link();

            // If on the main front page, add the network home title.
            elseif ( true === $this->args['show_title'] )
                $this->items[] = is_multisite() && true === $this->args['network'] ? get_bloginfo( 'name' ) : $this->labels['home'];
        }
    }

    /**
     * Adds items for the posts page (i.e., is_home()) to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_blog_items() {

        // Get the post ID and post.
        $travel_tourist_post_id = get_queried_object_id();
        $post    = get_post( $travel_tourist_post_id );

        // If the post has parents, add them to the trail.
        if ( 0 < $post->post_parent )
            $this->add_post_parents( $post->post_parent );

        // Get the page title.
        $travel_tourist_title = get_the_title( $travel_tourist_post_id );

        // Add the posts page item.
        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_permalink( $travel_tourist_post_id ) ), $travel_tourist_title );

        elseif ( $travel_tourist_title && true === $this->args['show_title'] )
            $this->items[] = $travel_tourist_title;
    }

    /**
     * Adds singular post items to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_singular_items() {

        // Get the queried post.
        $post    = get_queried_object();
        $travel_tourist_post_id = get_queried_object_id();

        // If the post has a parent, follow the parent trail.
        if ( 0 < $post->post_parent )
            $this->add_post_parents( $post->post_parent );

        // If the post doesn't have a parent, get its hierarchy based off the post type.
        else
            $this->add_post_hierarchy( $travel_tourist_post_id );

        // Display terms for specific post type taxonomy if requested.
        if ( ! empty( $this->post_taxonomy[ $post->post_type ] ) )
            $this->add_post_terms( $travel_tourist_post_id, $this->post_taxonomy[ $post->post_type ] );

        // End with the post title.
        if ( $travel_tourist_post_title = single_post_title( '', false ) ) {

            if ( ( 1 < get_query_var( 'page' ) || is_paged() ) || ( get_option( 'page_comments' ) && 1 < absint( get_query_var( 'cpage' ) ) ) )
                $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_permalink( $travel_tourist_post_id ) ), $travel_tourist_post_title );

            elseif ( true === $this->args['show_title'] )
                $this->items[] = $travel_tourist_post_title;
        }
    }

    /**
     * Adds the items to the trail items array for taxonomy term archives.
     *
     * @since  1.0.0
     * @access protected
     * @global object $wp_rewrite
     * @return void
     */
    protected function add_term_archive_items() {
        global $wp_rewrite;

        // Get some taxonomy and term variables.
        $travel_tourist_term           = get_queried_object();
        $travel_tourist_taxonomy       = get_taxonomy( $travel_tourist_term->taxonomy );
        $travel_tourist_done_post_type = false;

        // If there are rewrite rules for the taxonomy.
        if ( false !== $travel_tourist_taxonomy->rewrite ) {

            // If 'with_front' is true, dd $wp_rewrite->front to the trail.
            if ( isset( $travel_tourist_taxonomy->rewrite['with_front'] ) && $travel_tourist_taxonomy->rewrite['with_front'] && isset( $wp_rewrite->front ) && $wp_rewrite->front )
                $this->add_rewrite_front_items();

            // Get parent pages by path if they exist.
            $this->add_path_parents( $travel_tourist_taxonomy->rewrite['slug'] );

            // Add post type archive if its 'has_archive' matches the taxonomy rewrite 'slug'.
            if ( $travel_tourist_taxonomy->rewrite['slug'] ) {

                $travel_tourist_slug = trim( $travel_tourist_taxonomy->rewrite['slug'], '/' );

                // Deals with the situation if the slug has a '/' between multiple
                // strings. For example, "movies/genres" where "movies" is the post
                // type archive.
                $travel_tourist_matches = explode( '/', $travel_tourist_slug );

                // If matches are found for the path.
                if ( isset( $travel_tourist_matches ) ) {

                    // Reverse the array of matches to search for posts in the proper order.
                    $travel_tourist_matches = array_reverse( $travel_tourist_matches );

                    // Loop through each of the path matches.
                    foreach ( $travel_tourist_matches as $travel_tourist_match ) {

                        // If a match is found.
                        $travel_tourist_slug = $travel_tourist_match;

                        // Get public post types that match the rewrite slug.
                        $travel_tourist_post_types = $this->get_post_types_by_slug( $travel_tourist_match );

                        if ( ! empty( $travel_tourist_post_types ) ) {

                            $travel_tourist_post_type_object = $travel_tourist_post_types[0];

                            // Add support for a non-standard label of 'archive_title' (special use case).
                            $travel_tourist_label = ! empty( $travel_tourist_post_type_object->labels->archive_title ) ? $travel_tourist_post_type_object->labels->archive_title : $travel_tourist_post_type_object->labels->name;

                            // Core filter hook.
                            $travel_tourist_label = apply_filters( 'post_type_archive_title', $travel_tourist_label, $travel_tourist_post_type_object->name );

                            // Add the post type archive link to the trail.
                            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_post_type_archive_link( $travel_tourist_post_type_object->name ) ), $travel_tourist_label );

                            $travel_tourist_done_post_type = true;

                            // Break out of the loop.
                            break;
                        }
                    }
                }
            }
        }

        // If there's a single post type for the taxonomy, use it.
        if ( false === $travel_tourist_done_post_type && 1 === count( $travel_tourist_taxonomy->object_type ) && post_type_exists( $travel_tourist_taxonomy->object_type[0] ) ) {

            // If the post type is 'post'.
            if ( 'post' === $travel_tourist_taxonomy->object_type[0] ) {
                $travel_tourist_post_id = get_option( 'page_for_posts' );

                if ( 'posts' !== get_option( 'show_on_front' ) && 0 < $travel_tourist_post_id )
                    $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_permalink( $travel_tourist_post_id ) ), get_the_title( $travel_tourist_post_id ) );

            // If the post type is not 'post'.
            } else {
                $travel_tourist_post_type_object = get_post_type_object( $travel_tourist_taxonomy->object_type[0] );

                $travel_tourist_label = ! empty( $travel_tourist_post_type_object->labels->archive_title ) ? $travel_tourist_post_type_object->labels->archive_title : $travel_tourist_post_type_object->labels->name;

                // Core filter hook.
                $travel_tourist_label = apply_filters( 'post_type_archive_title', $travel_tourist_label, $travel_tourist_post_type_object->name );

                $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_post_type_archive_link( $travel_tourist_post_type_object->name ) ), $travel_tourist_label );
            }
        }

        // If the taxonomy is hierarchical, list its parent terms.
        if ( is_taxonomy_hierarchical( $travel_tourist_term->taxonomy ) && $travel_tourist_term->parent )
            $this->add_term_parents( $travel_tourist_term->parent, $travel_tourist_term->taxonomy );

        // Add the term name to the trail end.
        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_term_link( $travel_tourist_term, $travel_tourist_term->taxonomy ) ), single_term_title( '', false ) );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = single_term_title( '', false );
    }

    /**
     * Adds the items to the trail items array for post type archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_post_type_archive_items() {

        // Get the post type object.
        $travel_tourist_post_type_object = get_post_type_object( get_query_var( 'post_type' ) );

        if ( false !== $travel_tourist_post_type_object->rewrite ) {

            // If 'with_front' is true, add $wp_rewrite->front to the trail.
            if ( isset( $travel_tourist_post_type_object->rewrite['with_front'] ) && $travel_tourist_post_type_object->rewrite['with_front'] )
                $this->add_rewrite_front_items();

            // If there's a rewrite slug, check for parents.
            if ( ! empty( $travel_tourist_post_type_object->rewrite['slug'] ) )
                $this->add_path_parents( $travel_tourist_post_type_object->rewrite['slug'] );
        }

        // Add the post type [plural] name to the trail end.
        if ( is_paged() || is_author() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_post_type_archive_link( $travel_tourist_post_type_object->name ) ), post_type_archive_title( '', false ) );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = post_type_archive_title( '', false );

        // If viewing a post type archive by author.
        if ( is_author() )
            $this->add_user_archive_items();
    }

    /**
     * Adds the items to the trail items array for user (author) archives.
     *
     * @since  1.0.0
     * @access protected
     * @global object $wp_rewrite
     * @return void
     */
    protected function add_user_archive_items() {
        global $wp_rewrite;

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Get the user ID.
        $user_id = get_query_var( 'author' );

        // If $author_base exists, check for parent pages.
        if ( ! empty( $wp_rewrite->author_base ) && ! is_post_type_archive() )
            $this->add_path_parents( $wp_rewrite->author_base );

        // Add the author's display name to the trail end.
        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_author_posts_url( $user_id ) ), get_the_author_meta( 'display_name', $user_id ) );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = get_the_author_meta( 'display_name', $user_id );
    }

    /**
     * Adds the items to the trail items array for minute + hour archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_minute_hour_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Add the minute + hour item.
        if ( true === $this->args['show_title'] )
            $this->items[] = sprintf( $this->labels['archive_minute_hour'], get_the_time( esc_html_x( 'g:i a', 'minute and hour archives time format', 'travel-tourist' ) ) );
    }

    /**
     * Adds the items to the trail items array for minute archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_minute_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Add the minute item.
        if ( true === $this->args['show_title'] )
            $this->items[] = sprintf( $this->labels['archive_minute'], get_the_time( esc_html_x( 'i', 'minute archives time format', 'travel-tourist' ) ) );
    }

    /**
     * Adds the items to the trail items array for hour archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_hour_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Add the hour item.
        if ( true === $this->args['show_title'] )
            $this->items[] = sprintf( $this->labels['archive_hour'], get_the_time( esc_html_x( 'g a', 'hour archives time format', 'travel-tourist' ) ) );
    }

    /**
     * Adds the items to the trail items array for day archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_day_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Get year, month, and day.
        $travel_tourist_year  = sprintf( $this->labels['archive_year'],  get_the_time( esc_html_x( 'Y', 'yearly archives date format',  'travel-tourist' ) ) );
        $travel_tourist_month = sprintf( $this->labels['archive_month'], get_the_time( esc_html_x( 'F', 'monthly archives date format', 'travel-tourist' ) ) );
        $travel_tourist_day   = sprintf( $this->labels['archive_day'],   get_the_time( esc_html_x( 'j', 'daily archives date format',   'travel-tourist' ) ) );

        // Add the year and month items.
        $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_year_link( get_the_time( 'Y' ) ) ), $travel_tourist_year );
        $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ) ), $travel_tourist_month );

        // Add the day item.
        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_day_link( get_the_time( 'Y' ) ), get_the_time( 'm' ), get_the_time( 'd' ) ), $travel_tourist_day );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = $travel_tourist_day;
    }

    /**
     * Adds the items to the trail items array for week archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_week_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Get the year and week.
        $travel_tourist_year = sprintf( $this->labels['archive_year'],  get_the_time( esc_html_x( 'Y', 'yearly archives date format', 'travel-tourist' ) ) );
        $travel_tourist_week = sprintf( $this->labels['archive_week'],  get_the_time( esc_html_x( 'W', 'weekly archives date format', 'travel-tourist' ) ) );

        // Add the year item.
        $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_year_link( get_the_time( 'Y' ) ) ), $travel_tourist_year );

        // Add the week item.
        if ( is_paged() )
            $this->items[] = esc_url( get_archives_link( add_query_arg( array( 'm' => get_the_time( 'Y' ), 'w' => get_the_time( 'W' ) ), home_url() ), $travel_tourist_week, false ) );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = $travel_tourist_week;
    }

    /**
     * Adds the items to the trail items array for month archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_month_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Get the year and month.
        $travel_tourist_year  = sprintf( $this->labels['archive_year'],  get_the_time( esc_html_x( 'Y', 'yearly archives date format',  'travel-tourist' ) ) );
        $travel_tourist_month = sprintf( $this->labels['archive_month'], get_the_time( esc_html_x( 'F', 'monthly archives date format', 'travel-tourist' ) ) );

        // Add the year item.
        $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_year_link( get_the_time( 'Y' ) ) ), $travel_tourist_year );

        // Add the month item.
        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ) ), $travel_tourist_month );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = $travel_tourist_month;
    }

    /**
     * Adds the items to the trail items array for year archives.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_year_archive_items() {

        // Add $wp_rewrite->front to the trail.
        $this->add_rewrite_front_items();

        // Get the year.
        $travel_tourist_year  = sprintf( $this->labels['archive_year'],  get_the_time( esc_html_x( 'Y', 'yearly archives date format',  'travel-tourist' ) ) );

        // Add the year item.
        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_year_link( get_the_time( 'Y' ) ) ), $travel_tourist_year );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = $travel_tourist_year;
    }

    /**
     * Adds the items to the trail items array for archives that don't have a more specific method
     * defined in this class.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_default_archive_items() {

        // If this is a date-/time-based archive, add $wp_rewrite->front to the trail.
        if ( is_date() || is_time() )
            $this->add_rewrite_front_items();

        if ( true === $this->args['show_title'] )
            $this->items[] = $this->labels['archives'];
    }

    /**
     * Adds the items to the trail items array for search results.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_search_items() {

        if ( is_paged() )
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_search_link() ), sprintf( $this->labels['search'], get_search_query() ) );

        elseif ( true === $this->args['show_title'] )
            $this->items[] = sprintf( $this->labels['search'], get_search_query() );
    }

    /**
     * Adds the items to the trail items array for 404 pages.
     *
     * @since  1.0.0
     * @access protected
     * @return void
     */
    protected function add_404_items() {

        if ( true === $this->args['show_title'] )
            $this->items[] = $this->labels['error_404'];
    }

    /**
     * Adds a specific post's parents to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @param  int    $travel_tourist_post_id
     * @return void
     */
    protected function add_post_parents( $travel_tourist_post_id ) {
        $travel_tourist_parents = array();

        while ( $travel_tourist_post_id ) {

            // Get the post by ID.
            $post = get_post( $travel_tourist_post_id );

            // If we hit a page that's set as the front page, bail.
            if ( 'page' == $post->post_type && 'page' == get_option( 'show_on_front' ) && $travel_tourist_post_id == get_option( 'page_on_front' ) )
                break;

            // Add the formatted post link to the array of parents.
            $travel_tourist_parents[] = sprintf( '<a href="%s">%s</a>', esc_url( get_permalink( $travel_tourist_post_id ) ), get_the_title( $travel_tourist_post_id ) );

            // If there's no longer a post parent, break out of the loop.
            if ( 0 >= $post->post_parent )
                break;

            // Change the post ID to the parent post to continue looping.
            $travel_tourist_post_id = $post->post_parent;
        }

        // Get the post hierarchy based off the final parent post.
        $this->add_post_hierarchy( $travel_tourist_post_id );

        // Display terms for specific post type taxonomy if requested.
        if ( ! empty( $this->post_taxonomy[ $post->post_type ] ) )
            $this->add_post_terms( $travel_tourist_post_id, $this->post_taxonomy[ $post->post_type ] );

        // Merge the parent items into the items array.
        $this->items = array_merge( $this->items, array_reverse( $travel_tourist_parents ) );
    }

    /**
     * Adds a specific post's hierarchy to the items array.  The hierarchy is determined by post type's
     * rewrite arguments and whether it has an archive page.
     *
     * @since  1.0.0
     * @access protected
     * @param  int    $travel_tourist_post_id
     * @return void
     */
    protected function add_post_hierarchy( $travel_tourist_post_id ) {

        // Get the post type.
        $travel_tourist_post_type        = get_post_type( $travel_tourist_post_id );
        $travel_tourist_post_type_object = get_post_type_object( $travel_tourist_post_type );

        // If this is the 'post' post type, get the rewrite front items and map the rewrite tags.
        if ( 'post' === $travel_tourist_post_type ) {

            // Add $wp_rewrite->front to the trail.
            $this->add_rewrite_front_items();

            // Map the rewrite tags.
            $this->map_rewrite_tags( $travel_tourist_post_id, get_option( 'permalink_structure' ) );
        }

        // If the post type has rewrite rules.
        elseif ( false !== $travel_tourist_post_type_object->rewrite ) {

            // If 'with_front' is true, add $wp_rewrite->front to the trail.
            if ( isset( $travel_tourist_post_type_object->rewrite['with_front'] ) && $travel_tourist_post_type_object->rewrite['with_front'] )
                $this->add_rewrite_front_items();

            // If there's a path, check for parents.
            if ( ! empty( $travel_tourist_post_type_object->rewrite['slug'] ) )
                $this->add_path_parents( $travel_tourist_post_type_object->rewrite['slug'] );
        }

        // If there's an archive page, add it to the trail.
        if ( $travel_tourist_post_type_object->has_archive ) {

            // Add support for a non-standard label of 'archive_title' (special use case).
            $travel_tourist_label = ! empty( $travel_tourist_post_type_object->labels->archive_title ) ? $travel_tourist_post_type_object->labels->archive_title : $travel_tourist_post_type_object->labels->name;

            // Core filter hook.
            $travel_tourist_label = apply_filters( 'post_type_archive_title', $travel_tourist_label, $travel_tourist_post_type_object->name );

            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_post_type_archive_link( $travel_tourist_post_type ) ), $travel_tourist_label );
        }

        // Map the rewrite tags if there's a `%` in the slug.
        if ( 'post' !== $travel_tourist_post_type && ! empty( $travel_tourist_post_type_object->rewrite['slug'] ) && false !== strpos( $travel_tourist_post_type_object->rewrite['slug'], '%' ) )
            $this->map_rewrite_tags( $travel_tourist_post_id, $travel_tourist_post_type_object->rewrite['slug'] );
    }

    /**
     * Gets post types by slug.  This is needed because the get_post_types() function doesn't exactly
     * match the 'has_archive' argument when it's set as a string instead of a boolean.
     *
     * @since  0.6.0
     * @access protected
     * @param  int    $travel_tourist_slug  The post type archive slug to search for.
     * @return void
     */
    protected function get_post_types_by_slug( $travel_tourist_slug ) {

        $travel_tourist_return = array();

        $travel_tourist_post_types = get_post_types( array(), 'objects' );

        foreach ( $travel_tourist_post_types as $travel_tourist_type ) {

            if ( $travel_tourist_slug === $travel_tourist_type->has_archive || ( true === $travel_tourist_type->has_archive && $travel_tourist_slug === $travel_tourist_type->rewrite['slug'] ) )
                $travel_tourist_return[] = $travel_tourist_type;
        }

        return $travel_tourist_return;
    }

    /**
     * Adds a post's terms from a specific taxonomy to the items array.
     *
     * @since  1.0.0
     * @access protected
     * @param  int     $travel_tourist_post_id  The ID of the post to get the terms for.
     * @param  string  $travel_tourist_taxonomy The taxonomy to get the terms from.
     * @return void
     */
    protected function add_post_terms( $travel_tourist_post_id, $travel_tourist_taxonomy ) {

        // Get the post type.
        $travel_tourist_post_type = get_post_type( $travel_tourist_post_id );

        // Get the post categories.
        $travel_tourist_terms = get_the_terms( $travel_tourist_post_id, $travel_tourist_taxonomy );

        // Check that categories were returned.
        if ( $travel_tourist_terms && ! is_wp_error( $travel_tourist_terms ) ) {

            // Sort the terms by ID and get the first category.
            if ( function_exists( 'wp_list_sort' ) )
                $travel_tourist_terms = wp_list_sort( $travel_tourist_terms, 'term_id' );

            else
                usort( $travel_tourist_terms, '_usort_terms_by_ID' );

            $travel_tourist_term = get_term( $travel_tourist_terms[0], $travel_tourist_taxonomy );

            // If the category has a parent, add the hierarchy to the trail.
            if ( 0 < $travel_tourist_term->parent )
                $this->add_term_parents( $travel_tourist_term->parent, $travel_tourist_taxonomy );

            // Add the category archive link to the trail.
            $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_term_link( $travel_tourist_term, $travel_tourist_taxonomy ) ), $travel_tourist_term->name );
        }
    }

    /**
     * Get parent posts by path.  Currently, this method only supports getting parents of the 'page'
     * post type.  The goal of this function is to create a clear path back to home given what would
     * normally be a "ghost" directory.  If any page matches the given path, it'll be added.
     *
     * @since  1.0.0
     * @access protected
     * @param  string $travel_tourist_path The path (slug) to search for posts by.
     * @return void
     */
    function add_path_parents( $travel_tourist_path ) {

        // Trim '/' off $travel_tourist_path in case we just got a simple '/' instead of a real path.
        $travel_tourist_path = trim( $travel_tourist_path, '/' );

        // If there's no path, return.
        if ( empty( $travel_tourist_path ) )
            return;

        // Get parent post by the path.
        $post = get_page_by_path( $travel_tourist_path );

        if ( ! empty( $post ) ) {
            $this->add_post_parents( $post->ID );
        }

        elseif ( is_null( $post ) ) {

            // Separate post names into separate paths by '/'.
            $travel_tourist_path = trim( $travel_tourist_path, '/' );
            preg_match_all( "/\/.*?\z/", $travel_tourist_path, $travel_tourist_matches );

            // If matches are found for the path.
            if ( isset( $travel_tourist_matches ) ) {

                // Reverse the array of matches to search for posts in the proper order.
                $travel_tourist_matches = array_reverse( $travel_tourist_matches );

                // Loop through each of the path matches.
                foreach ( $travel_tourist_matches as $travel_tourist_match ) {

                    // If a match is found.
                    if ( isset( $travel_tourist_match[0] ) ) {

                        // Get the parent post by the given path.
                        $travel_tourist_path = str_replace( $travel_tourist_match[0], '', $travel_tourist_path );
                        $post = get_page_by_path( trim( $travel_tourist_path, '/' ) );

                        // If a parent post is found, set the $travel_tourist_post_id and break out of the loop.
                        if ( ! empty( $post ) && 0 < $post->ID ) {
                            $this->add_post_parents( $post->ID );
                            break;
                        }
                    }
                }
            }
        }
    }

    /**
     * Searches for term parents of hierarchical taxonomies.  This function is similar to the WordPress
     * function get_category_parents() but handles any type of taxonomy.
     *
     * @since  1.0.0
     * @param  int    $travel_tourist_term_id  ID of the term to get the parents of.
     * @param  string $travel_tourist_taxonomy Name of the taxonomy for the given term.
     * @return void
     */
    function add_term_parents( $travel_tourist_term_id, $travel_tourist_taxonomy ) {

        // Set up some default arrays.
        $travel_tourist_parents = array();

        // While there is a parent ID, add the parent term link to the $travel_tourist_parents array.
        while ( $travel_tourist_term_id ) {

            // Get the parent term.
            $travel_tourist_term = get_term( $travel_tourist_term_id, $travel_tourist_taxonomy );

            // Add the formatted term link to the array of parent terms.
            $travel_tourist_parents[] = sprintf( '<a href="%s">%s</a>', esc_url( get_term_link( $travel_tourist_term, $travel_tourist_taxonomy ) ), $travel_tourist_term->name );

            // Set the parent term's parent as the parent ID.
            $travel_tourist_term_id = $travel_tourist_term->parent;
        }

        // If we have parent terms, reverse the array to put them in the proper order for the trail.
        if ( ! empty( $travel_tourist_parents ) )
            $this->items = array_merge( $this->items, array_reverse( $travel_tourist_parents ) );
    }

    /**
     * Turns %tag% from permalink structures into usable links for the breadcrumb trail.  This feels kind of
     * hackish for now because we're checking for specific %tag% examples and only doing it for the 'post'
     * post type.  In the future, maybe it'll handle a wider variety of possibilities, especially for custom post
     * types.
     *
     * @since  0.6.0
     * @access protected
     * @param  int    $travel_tourist_post_id ID of the post whose parents we want.
     * @param  string $travel_tourist_path    Path of a potential parent page.
     * @param  array  $travel_tourist_args    Mixed arguments for the menu.
     * @return array
     */
    protected function map_rewrite_tags( $travel_tourist_post_id, $travel_tourist_path ) {

        $post = get_post( $travel_tourist_post_id );

        // Trim '/' from both sides of the $travel_tourist_path.
        $travel_tourist_path = trim( $travel_tourist_path, '/' );

        // Split the $travel_tourist_path into an array of strings.
        $travel_tourist_matches = explode( '/', $travel_tourist_path );

        // If matches are found for the path.
        if ( is_array( $travel_tourist_matches ) ) {

            // Loop through each of the matches, adding each to the $trail array.
            foreach ( $travel_tourist_matches as $travel_tourist_match ) {

                // Trim any '/' from the $travel_tourist_match.
                $travel_tourist_tag = trim( $travel_tourist_match, '/' );

                // If using the %year% tag, add a link to the yearly archive.
                if ( '%year%' == $travel_tourist_tag )
                    $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_year_link( get_the_time( 'Y', $travel_tourist_post_id ) ) ), sprintf( $this->labels['archive_year'], get_the_time( esc_html_x( 'Y', 'yearly archives date format',  'travel-tourist' ) ) ) );

                // If using the %monthnum% tag, add a link to the monthly archive.
                elseif ( '%monthnum%' == $travel_tourist_tag )
                    $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_month_link( get_the_time( 'Y', $travel_tourist_post_id ), get_the_time( 'm', $travel_tourist_post_id ) ) ), sprintf( $this->labels['archive_month'], get_the_time( esc_html_x( 'F', 'monthly archives date format', 'travel-tourist' ) ) ) );

                // If using the %day% tag, add a link to the daily archive.
                elseif ( '%day%' == $travel_tourist_tag )
                    $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_day_link( get_the_time( 'Y', $travel_tourist_post_id ), get_the_time( 'm', $travel_tourist_post_id ), get_the_time( 'd', $travel_tourist_post_id ) ) ), sprintf( $this->labels['archive_day'], get_the_time( esc_html_x( 'j', 'daily archives date format', 'travel-tourist' ) ) ) );

                // If using the %author% tag, add a link to the post author archive.
                elseif ( '%author%' == $travel_tourist_tag )
                    $this->items[] = sprintf( '<a href="%s">%s</a>', esc_url( get_author_posts_url( $post->post_author ) ), get_the_author_meta( 'display_name', $post->post_author ) );

                // If using the %category% tag, add a link to the first category archive to match permalinks.
                elseif ( taxonomy_exists( trim( $travel_tourist_tag, '%' ) ) ) {

                    // Force override terms in this post type.
                    $this->post_taxonomy[ $post->post_type ] = false;

                    // Add the post categories.
                    $this->add_post_terms( $travel_tourist_post_id, trim( $travel_tourist_tag, '%' ) );
                }
            }
        }
    }
}