<?php
/**
 *
 * This file is part of phpFastCache.
 *
 * @license MIT License (MIT)
 *
 * For full copyright and license information, please see the docs/CREDITS.txt file.
 *
 * @author Khoa Bui (khoaofgod)  <khoaofgod@gmail.com> http://www.phpfastcache.com
 * @author Georges.L (Geolim4)  <contact@geolim4.com>
 *
 */

namespace phpFastCache\Drivers\Redis;

use phpFastCache\Core\Pool\DriverBaseTrait;
use phpFastCache\Core\Pool\ExtendedCacheItemPoolInterface;
use phpFastCache\Entities\DriverStatistic;
use phpFastCache\Exceptions\phpFastCacheDriverCheckException;
use phpFastCache\Exceptions\phpFastCacheDriverException;
use phpFastCache\Exceptions\phpFastCacheInvalidArgumentException;
use phpFastCache\Exceptions\phpFastCacheLogicException;
use Psr\Cache\CacheItemInterface;
use Redis as RedisClient;

/**
 * Class Driver
 * @package phpFastCache\Drivers
 * @property RedisClient $instance Instance of driver service
 */
class Driver implements ExtendedCacheItemPoolInterface
{
    use DriverBaseTrait;

    /**
     * Driver constructor.
     * @param array $config
     * @throws phpFastCacheDriverException
     */
    public function __construct(array $config = [])
    {
        $this->setup($config);

        if (!$this->driverCheck()) {
            throw new phpFastCacheDriverCheckException(sprintf(self::DRIVER_CHECK_FAILURE, $this->getDriverName()));
        } else {
            $this->driverConnect();
        }
    }

    /**
     * @return bool
     */
    public function driverCheck()
    {
        return extension_loaded('Redis');
    }

    /**
     * @param \Psr\Cache\CacheItemInterface $item
     * @return mixed
     * @throws phpFastCacheInvalidArgumentException
     */
    protected function driverWrite(CacheItemInterface $item)
    {
        /**
         * Check for Cross-Driver type confusion
         */
        if ($item instanceof Item) {
            $ttl = $item->getExpirationDate()->getTimestamp() - time();

            /**
             * @see https://redis.io/commands/setex
             * @see https://redis.io/commands/expire
             */
            if($ttl <= 0){
                return $this->instance->expire($item->getKey(), 0);
            }else{
                return $this->instance->setex($item->getKey(), $ttl, $this->encode($this->driverPreWrap($item)));
            }
        } else {
            throw new phpFastCacheInvalidArgumentException('Cross-Driver type confusion detected');
        }
    }

    /**
     * @param \Psr\Cache\CacheItemInterface $item
     * @return null|array
     */
    protected function driverRead(CacheItemInterface $item)
    {
        $val = $this->instance->get($item->getKey());
        if ($val == false) {
            return null;
        } else {
            return $this->decode($val);
        }
    }

    /**
     * @param \Psr\Cache\CacheItemInterface $item
     * @return bool
     * @throws phpFastCacheInvalidArgumentException
     */
    protected function driverDelete(CacheItemInterface $item)
    {
        /**
         * Check for Cross-Driver type confusion
         */
        if ($item instanceof Item) {
            return $this->instance->del($item->getKey());
        } else {
            throw new phpFastCacheInvalidArgumentException('Cross-Driver type confusion detected');
        }
    }

    /**
     * @return bool
     */
    protected function driverClear()
    {
        return $this->instance->flushDB();
    }

    /**
     * @return bool
     * @throws phpFastCacheLogicException
     */
    protected function driverConnect()
    {
        if ($this->instance instanceof RedisClient) {
            throw new phpFastCacheLogicException('Already connected to Redis server');
        } else {
            $this->instance = $this->instance ?: new RedisClient();

            $host = isset($this->config[ 'host' ]) ? (string) $this->config[ 'host' ] : '127.0.0.1';
            $path = isset($this->config[ 'path' ]) ? (string) $this->config[ 'path' ] : false;
            $port = isset($this->config[ 'port' ]) ? (int) $this->config[ 'port' ] : 6379;
            $password = isset($this->config[ 'password' ]) ? (string) $this->config[ 'password' ] : '';
            $database = isset($this->config[ 'database' ]) ?  $this->config[ 'database' ] : false;
            $timeout = isset($this->config[ 'timeout' ]) ?  $this->config[ 'timeout' ] : '';

            /**
             * If path is provided we consider it as an UNIX Socket
             */
            if($path){
                $isConnected = $this->instance->connect($path);
            }else{
                $isConnected = $this->instance->connect($host, (int)$port, (int)$timeout);
            }

            if (!$isConnected && $path) {
                return false;
            } else if(!$path) {
                if ($password && !$this->instance->auth($password)) {
                    return false;
                }
            }
            if ($database !== false) {
                $this->instance->select((int)$database);
            }
            return true;
        }
    }

    /********************
     *
     * PSR-6 Extended Methods
     *
     *******************/

    /**
     * @return DriverStatistic
     */
    public function getStats()
    {
        // used_memory
        $info = $this->instance->info();
        $date = (new \DateTime())->setTimestamp(time() - $info[ 'uptime_in_seconds' ]);

        return (new DriverStatistic())
          ->setData(implode(', ', array_keys($this->itemInstances)))
          ->setRawData($info)
          ->setSize($info[ 'used_memory' ])
          ->setInfo(sprintf("The Redis daemon v%s is up since %s.\n For more information see RawData. \n Driver size includes the memory allocation size.",
            $info[ 'redis_version' ], $date->format(DATE_RFC2822)));
    }
}